<?php

/*
 * This file is part of the Claroline Connect package.
 *
 * (c) Claroline Consortium <consortium@claroline.net>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Claroline\InstallationBundle\Manager;

use Claroline\BundleRecorder\Log\LoggableTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\ContainerAwareInterface;
use Claroline\CoreBundle\Entity\Bundle;
use Claroline\MigrationBundle\Manager\Manager;
use Claroline\MigrationBundle\Migrator\Migrator;
use Claroline\InstallationBundle\Fixtures\FixtureLoader;
use Claroline\InstallationBundle\Bundle\BundleVersion;
use Claroline\InstallationBundle\Bundle\InstallableInterface;
use Claroline\InstallationBundle\Bundle\InstallableBundle;
use Claroline\InstallationBundle\Additional\AdditionalInstallerInterface;

class InstallationManager
{
    use LoggableTrait;

    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * @var string
     */
    private $environment;

    /**
     * @var Manager
     */
    private $migrationManager;

    /**
     * @var FixtureLoader
     */
    private $fixtureLoader;

    public function __construct(
        ContainerInterface $container,
        Manager $migrationManager,
        FixtureLoader $fixtureLoader,
        $environment
    )
    {
        $this->container = $container;
        $this->migrationManager = $migrationManager;
        $this->fixtureLoader = $fixtureLoader;
        $this->environment = $environment;
    }

    public function install(InstallableInterface $bundle, $requiredOnly = true)
    {
        $this->log(sprintf("<comment>Installing %s...</comment>", $bundle->getName()));
        $additionalInstaller = $this->getAdditionalInstaller($bundle);

        if ($additionalInstaller) {
            $this->log('Launching pre-installation actions...');
            $additionalInstaller->preInstall();
        }

        if ($bundle->hasMigrations()) {
            $this->log('Executing migrations...');
            $this->migrationManager->upgradeBundle($bundle, Migrator::VERSION_FARTHEST);
        }

        if ($fixturesDir = $bundle->getRequiredFixturesDirectory($this->environment)) {
            $this->log('Loading required fixtures...');
            $this->fixtureLoader->load($bundle, $fixturesDir);
        }

        if (!$requiredOnly && $fixturesDir = $bundle->getOptionalFixturesDirectory($this->environment)) {
            $this->log('Loading optional fixtures...');
            $this->fixtureLoader->load($bundle, $fixturesDir);
        }

        if ($additionalInstaller) {
            $this->log('Launching post-installation actions...');
            $additionalInstaller->postInstall();
        }

        $this->persistBundleInfo($bundle);
    }

    public function update(InstallableInterface $bundle, $currentVersion, $targetVersion)
    {
        $this->log(sprintf("<comment>Updating %s...</comment>", $bundle->getName()));
        $additionalInstaller = $this->getAdditionalInstaller($bundle);

        if ($additionalInstaller) {
            $this->log('Launching pre-update actions...');
            $additionalInstaller->preUpdate($currentVersion, $targetVersion);
        }

        if ($bundle->hasMigrations()) {
            $this->log('Executing migrations...');
            $this->migrationManager->upgradeBundle($bundle, Migrator::VERSION_FARTHEST);
        }

        if ($additionalInstaller) {
            $this->log('Launching post-update actions...');
            $additionalInstaller->postUpdate($currentVersion, $targetVersion);
        }

        $this->persistBundleInfo($bundle);
    }

    public function uninstall(InstallableInterface $bundle)
    {
        $this->log(sprintf("<comment>Uninstalling %s...</comment>", $bundle->getName()));
        $additionalInstaller = $this->getAdditionalInstaller($bundle);

        if ($additionalInstaller) {
            $this->log('Launching pre-uninstallation actions...');
            $additionalInstaller->preUninstall();
        }

        if ($bundle->hasMigrations()) {
            $this->log('Executing migrations...');
            $this->migrationManager->downgradeBundle($bundle, Migrator::VERSION_FARTHEST);
        }

        if ($additionalInstaller) {
            $this->log('Launching post-uninstallation actions...');
            $additionalInstaller->postUninstall();
        }
    }

    /**
     * @param InstallableInterface $bundle
     *
     * @return AdditionalInstallerInterface|bool
     */
    private function getAdditionalInstaller(InstallableInterface $bundle)
    {
        $installer = $bundle->getAdditionalInstaller();

        if ($installer instanceof AdditionalInstallerInterface) {
            $installer->setEnvironment($this->environment);
            $installer->setLogger($this->logger);

            if ($installer instanceof ContainerAwareInterface) {
                $installer->setContainer($this->container);
            }

            return $installer;
        }

        return false;
    }

    private function persistBundleInfo(InstallableInterface $bundle)
    {
        if ($bundle instanceof InstallableBundle) {
            $ds = DIRECTORY_SEPARATOR;
            $om = $this->container->get('doctrine.orm.entity_manager');
            $entity = $om->getRepository('ClarolineCoreBundle:Bundle')
                ->findOneByName($bundle->getClarolineName());
    
            if (!$entity) {
                $entity = new Bundle();
            }
    
            $entity->setName($bundle->getClarolineName());
            $entity->setVersion($bundle->getVersion());
            $entity->setAuthors($bundle->getAuthors());
            $entity->setType($bundle->getType());
            $entity->setDescription($bundle->getDescription());
            $entity->setLicense($bundle->getLicense());
            $entity->setTargetDir($bundle->getTargetDir());
            $entity->setBasePath($bundle->getBasePath());
            $om->persist($entity);
            $this->log(sprintf("Updating %s info...", $bundle->getName()));
            $om->flush();
        }
    }
}
